/**
  ******************************************************************************
  * @file    main.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   Program entry point
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 

/* Includes ------------------------------------------------------------------*/
#include <string.h>   // fr memcpy

#include "main.h"
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

#define MAXREADERSFORTEST 6

static uint32_t g_WritersActive,g_ReadersActive,g_ReaderXActCt;
static xSemaphoreHandle g_ReaderMutex;

#define READERTASKPRIORITY (configMAX_PRIORITIES-1)
#define WRITERTASKPRIORITY (configMAX_PRIORITIES-1)


/** @brief binds CPU time from the task
 *
 *  can be configure to wait busy or cooperatively
 *
 *  @param p_Delay Simulated time the writer is in the lock.
 *  @return none
 */

unsigned long TaskUseTime(unsigned long p_Delay)
{
#ifndef COOPERATIVEWAIT
    while (p_Delay--)
        vTaskDelay(2);
#else
    TickType_t a_Future = xTaskGetTickCount() + p_Delay;
    // won't work well on architectures where portTICK_TYPE_IS_ATOMIC is not defined
    while (xTaskGetTickCount()<a_Future);

#endif
}

/** @brief Simulates a critical writer operation (under control of the writer side of the rw lock)
 *
 *  As a side effect, turns on the red LED if a reader is present (visualizes lock violation)
 *
 *  @param p_Delay Simulated time the writer is in the lock.
 *  @return none
 */

void DoWriterLoop(unsigned long p_Delay)
{
    g_WritersActive = 1;
    STM_EVAL_LEDToggle(LED_BLUE);    
    if (g_ReadersActive)
    {
        STM_EVAL_LEDOn(LED_RED);
    }
    TaskUseTime (p_Delay);
    g_WritersActive = 0;
  
}

/** @brief Writer task function
 *
 *
 *  @param p_Dummy required FreeRTOS task parameter. No op for the writer.
 *  @return none
 */

void WriterTask(void *p_Dummy)
{
    while(1)
    {
        unsigned long a_DelayThisTime = /*rand()%100*/2;
        ClaimRwLock(LOCK_WRITER);
        DoWriterLoop(a_DelayThisTime);
        ReleaseRWLock(LOCK_WRITER);
        vTaskDelay(/*rand()%2000*/2);
    }
}

/** @brief Simulates a critical reader operation (under control of the reader side of the rw lock)
 *
 *  As a side effect, turns on the orange LED if a writer is present (visualizes lock violation)
 *
 *  @param p_TaskId distinguishes between readers.
 *  @return none
 */

void DoReaderLoop(unsigned long p_TaskId,unsigned long theDelay)
{
    xSemaphoreTake(g_ReaderMutex,portMAX_DELAY); // this simply shields g_ReadersActive from being trashed through concurrent access
    g_ReadersActive |= (1 << p_TaskId);
    xSemaphoreGive(g_ReaderMutex);
    if (g_WritersActive)
    {
        STM_EVAL_LEDOn(LED_ORANGE);
    }
    TaskUseTime(theDelay);
    xSemaphoreTake(g_ReaderMutex,portMAX_DELAY);
    STM_EVAL_LEDToggle(LED_GREEN);    
    g_ReadersActive &= ~(1 << p_TaskId);
    xSemaphoreGive(g_ReaderMutex);
}

/** @brief Reader task function
 *
 *
 *  @param p_TaskId required FreeRTOS task parameter. Here distinguishes between readers.
 *  @return none
 */

void ReaderTask(void *p_TaskId)
{
    while(1)
    {
        unsigned long aDelayThisTime = rand()%100;
        ClaimRwLock(LOCK_READER);
        DoReaderLoop((unsigned long)p_TaskId,aDelayThisTime);
        ReleaseRWLock(LOCK_READER_TERMINAL);
        vTaskDelay(rand()%200);
    }
}

/** @brief Application entry point
 *
 *
 *  @return Dummy (required by unused C Runtime Code). Function does not return unless vTaskStartScheduler() has a problem.
 */

int main(void)
{
  unsigned long a_Loop;
  g_WritersActive = g_ReadersActive = 0;
  g_ReaderMutex = xSemaphoreCreateMutex();
  /*!< At this stage the microcontroller clock setting is already configured, 
       this is done through SystemInit() function which is called from startup
       file (startup_stm32f4xx.s) before to branch to application main.
       To reconfigure the default setting of SystemInit() function, refer to
        system_stm32f4xx.c file
     */     
       
  /* Initialize Leds mounted on STM32F4-Discovery board */
  STM_EVAL_LEDInit(LED_GREEN);
  STM_EVAL_LEDInit(LED_ORANGE);
  STM_EVAL_LEDInit(LED_BLUE);
  STM_EVAL_LEDInit(LED_RED);

  /* Setup SysTick Timer for 1 msec interrupts.
     ------------------------------------------
    1. The SysTick_Config() function is a CMSIS function which configure:
       - The SysTick Reload register with value passed as function parameter.
       - Configure the SysTick IRQ priority to the lowest value (0x0F).
       - Reset the SysTick Counter register.
       - Configure the SysTick Counter clock source to be Core Clock Source (HCLK).
       - Enable the SysTick Interrupt.
       - Start the SysTick Counter.
    
    2. You can change the SysTick Clock source to be HCLK_Div8 by calling the
       SysTick_CLKSourceConfig(SysTick_CLKSource_HCLK_Div8) just after the
       SysTick_Config() function call. The SysTick_CLKSourceConfig() is defined
       inside the misc.c file.

    3. You can change the SysTick IRQ priority by calling the
       NVIC_SetPriority(SysTick_IRQn,...) just after the SysTick_Config() function 
       call. The NVIC_SetPriority() is defined inside the core_cm4.h file.

    4. To adjust the SysTick time base, use the following formula:
                            
         Reload Value = SysTick Counter Clock (Hz) x  Desired Time base (s)
    
       - Reload Value is the parameter to be passed for SysTick_Config() function
       - Reload Value should not exceed 0xFFFFFF
   */
  if (SysTick_Config(SystemCoreClock / 1000))
  { 
    /* Capture error */ 
    while (1);
  }

  InitGlobalRWLock();  
  
  for (a_Loop=0;a_Loop<MAXREADERSFORTEST;a_Loop++)
  {
      unsigned char aTaskName[10];
      // normally we would use sprintf() here to generate a dynamic readable task name; however, sprintf() is terrible in several respects
      // for deeply embedded systems (see the book for details). For once it uses a lot of stack space, and second, it pulls in the allocation routines
      // which don't go well with our allocator. Here we make the assumption that a_Loop never exceeds 9!
      memcpy(aTaskName,"RTask00\0",8);
      aTaskName[6] = a_Loop + '0'; 
	        xTaskCreate(ReaderTask,aTaskName, configMINIMAL_STACK_SIZE,
			    (void *)a_Loop, READERTASKPRIORITY-(1+(a_Loop&1)), (TaskHandle_t *) 0);
  }
    
	xTaskCreate(WriterTask,"WriterTask", configMINIMAL_STACK_SIZE,
			    0, WRITERTASKPRIORITY, (TaskHandle_t *) 0);
  
/* Start the scheduler */
	vTaskStartScheduler();
}
